from mongoengine import Document, StringField, DateTimeField, BooleanField, IntField
from datetime import datetime
import uuid


class Client(Document):
    """Client model for client management"""
    meta = {'collection': 'clients'}

    # Primary fields
    client_id = StringField(required=True, unique=True, default=lambda: str(uuid.uuid4()))
    client_name = StringField(required=True, unique=True)

    # Contact information
    email = StringField()
    phone = StringField()
    address = StringField()

    # Additional details
    description = StringField()
    contact_person = StringField()

    # Statistics (will be computed)
    active_cases_count = IntField(default=0)
    inactive_cases_count = IntField(default=0)
    total_cases_count = IntField(default=0)

    # Soft delete
    is_deleted = BooleanField(default=False)
    deleted_at = DateTimeField()
    deleted_by = StringField()

    # Timestamps
    created_at = DateTimeField(default=datetime.utcnow)
    updated_at = DateTimeField(default=datetime.utcnow)
    created_by = StringField()
    updated_by = StringField()

    def __repr__(self):
        return f'<Client {self.client_name}>'

    def update_case_counts(self):
        """Update case statistics for this client"""
        from app.models.case import Case

        # Count active cases
        self.active_cases_count = Case.objects(
            client_id=self.client_id,
            status='active',
            is_deleted=False
        ).count()

        # Count inactive cases
        self.inactive_cases_count = Case.objects(
            client_id=self.client_id,
            status__in=['inactive', 'closed', 'pending'],
            is_deleted=False
        ).count()

        # Total cases
        self.total_cases_count = Case.objects(
            client_id=self.client_id,
            is_deleted=False
        ).count()

        self.save()
