from flask import request, jsonify
from flask_jwt_extended import create_access_token, create_refresh_token, jwt_required, get_jwt_identity
from app.api.v1 import api_v1_bp
from app.models.user import User
from app.models.session_log import SessionLog
from app.models.action_log import ActionLog
from app.models.refresh_token import RefreshToken
from app.utils.response_helper import success_response, error_response
from datetime import datetime, timedelta
import uuid


@api_v1_bp.route('/login', methods=['POST'])
def login():
    """
    User Login Endpoint
    ---
    tags:
      - Authentication
    parameters:
      - name: body
        in: body
        required: true
        description: Login credentials and device information
        schema:
          type: object
          required:
            - email
            - password
            - user_type
          properties:
            email:
              type: string
              example: "user@example.com"
            password:
              type: string
              example: "securePassword123"
            user_type:
              type: string
              enum: ["US", "AD"]
              example: "US"
              description: "User type - US (User) or AD (Admin)"
            timezone:
              type: string
              example: "Asia/Kolkata"
            device_type:
              type: string
              example: "mobile"
            device_fingerprint:
              type: string
              example: "abc123xyz789"
            mac_id:
              type: string
              example: "00:11:22:33:44:55"
            location:
              type: object
              properties:
                lat:
                  type: number
                  example: 28.6139
                lng:
                  type: number
                  example: 77.2090
      - name: X-Real-IP
        in: header
        type: string
        required: false
        description: Client IP address
      - name: User-Agent
        in: header
        type: string
        required: false
        description: Client user agent
    responses:
      200:
        description: Login successful
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "Login successful"
            data:
              type: object
              properties:
                user_id:
                  type: string
                access_token:
                  type: string
                refresh_token:
                  type: string
                session_log_id:
                  type: string
                expires_in:
                  type: integer
                  example: 3600
                user:
                  type: object
                  properties:
                    email:
                      type: string
                    full_name:
                      type: string
                    role:
                      type: string
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
            error:
              type: object
              properties:
                code:
                  type: string
                details:
                  type: string
            meta:
              type: object
      401:
        description: Authentication failed
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 401
            message:
              type: string
              example: "Invalid credentials"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "AUTH_FAILED"
                details:
                  type: string
                  example: "Email or password is incorrect"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get request data
        data = request.get_json()

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Validate required fields
        email = data.get('email')
        password = data.get('password')
        user_type = data.get('user_type')

        if not email or not password:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='login',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'Missing email or password'}
            ).save()

            return error_response(
                message="Email and password are required",
                error_code="VALIDATION_ERROR",
                error_details="Missing email or password",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by email and user_type (check if exists first)
        user = User.objects(email=email, user_type=user_type).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='login',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email, "user_type": user_type},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Invalid credentials",
                error_code="AUTH_FAILED",
                error_details="Email or password is incorrect",
                status_code=401,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='login',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email, "user_type": user_type},
                response_status='failed',
                error_details={'reason': 'User account is deactivated'}
            ).save()

            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated. Please contact administrator.",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Verify password (no encryption - direct comparison as per requirement)
        if user.password != password:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='login',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'Invalid password'}
            ).save()

            return error_response(
                message="Invalid credentials",
                error_code="AUTH_FAILED",
                error_details="Email or password is incorrect",
                status_code=401,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Generate tokens
        access_token = create_access_token(identity=user.user_id)
        refresh_token = create_refresh_token(identity=user.user_id)

        # Store refresh token in database
        import json
        location_data = data.get('location')
        refresh_token_record = RefreshToken.create_refresh_token(
            user_id=user.user_id,
            device_type=data.get('device_type'),
            device_fingerprint=data.get('device_fingerprint'),
            mac_id=data.get('mac_id'),
            location=json.dumps(location_data) if location_data else None,
            user_agent=user_agent,
            ip_address=ip_address,
            expires_in_days=30  # 30 days validity
        )
        # Update the refresh token string to match JWT refresh token
        refresh_token_record.refresh_token = refresh_token
        refresh_token_record.save()

        # Create session log
        session_log_id = str(uuid.uuid4())
        expires_at = datetime.utcnow() + timedelta(seconds=3600)

        session_log = SessionLog(
            session_log_id=session_log_id,
            user_id=user.user_id,
            access_token=access_token,
            refresh_token=refresh_token,
            device_type=data.get('device_type'),
            device_fingerprint=data.get('device_fingerprint'),
            mac_id=data.get('mac_id'),
            ip_address=ip_address,
            user_agent=user_agent,
            location=data.get('location'),
            timezone=data.get('timezone'),
            is_active=True,
            expires_at=expires_at
        )
        session_log.save()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='login',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={"email": email, "user_type": user_type},
            response_status='success'
        ).save()

        # Prepare response data
        response_data = {
            "user_id": user.user_id,
            "access_token": access_token,
            "refresh_token": refresh_token,
            "session_log_id": session_log_id,
            "expires_in": 3600,
            "user": {
                "email": user.email,
                "full_name": user.full_name,
                "role": user.user_type
            }
        }

        return success_response(
            message="Login successful",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            action_type='login',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/auth/forgot-password', methods=['POST'])
def forgot_password():
    """
    Forgot Password - Send OTP
    ---
    tags:
      - Authentication
    parameters:
      - name: body
        in: body
        required: true
        description: Forgot password request data
        schema:
          type: object
          required:
            - email
          properties:
            email:
              type: string
              example: "user@example.com"
              description: User's email address
            device_type:
              type: string
              example: "desktop"
              description: Device type (desktop, mobile, tablet)
            device_fingerprint:
              type: string
              example: "xyz789abc123"
              description: Unique device fingerprint
            mac_id:
              type: string
              example: "192.168.1.100"
              description: MAC ID or IP address
            location:
              type: object
              properties:
                lat:
                  type: number
                  example: 28.6139
                  description: Latitude
                lng:
                  type: number
                  example: 77.2090
                  description: Longitude
      - name: X-Real-IP
        in: header
        type: string
        required: false
        description: Client IP address
      - name: User-Agent
        in: header
        type: string
        required: false
        description: Client user agent
    responses:
      200:
        description: OTP sent successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "OTP sent successfully"
            data:
              type: object
              properties:
                otp_sent_to:
                  type: string
                  example: "user@example.com"
                otp_validity_minutes:
                  type: integer
                  example: 10
                  description: OTP validity in minutes
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      404:
        description: Email not found
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 404
            message:
              type: string
              example: "Email not found"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "EMAIL_NOT_FOUND"
                details:
                  type: string
                  example: "No user found with this email"
            meta:
              type: object
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Validation error"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
                  example: "Email is required"
            meta:
              type: object
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get request data
        data = request.get_json()

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Validate required fields
        email = data.get('email', '').strip().lower()

        if not email:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='forgot_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'Email is required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Email is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Validate email format
        if '@' not in email:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='forgot_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'Invalid email format'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Invalid email format",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by email (check if exists first)
        user = User.objects(email=email).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='forgot_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Email not found",
                error_code="EMAIL_NOT_FOUND",
                error_details="No user found with this email",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='forgot_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'User account is deactivated'}
            ).save()

            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated. Please contact administrator.",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Import OTP model
        from app.models.otp import OTP

        # Expire any existing OTPs for this user
        existing_otps = OTP.objects(user_id=user.user_id, otp_type='forgot_password', is_used=False, is_expired=False)
        for otp in existing_otps:
            otp.is_expired = True
            otp.save()

        # Generate OTP
        otp_code = OTP.generate_otp_code(6)
        otp_validity_minutes = 10

        # Create OTP record
        new_otp = OTP(
            user_id=user.user_id,
            email=user.email,
            otp_code=otp_code,
            otp_type='forgot_password',
            expires_at=datetime.utcnow() + timedelta(minutes=otp_validity_minutes),
            device_type=data.get('device_type'),
            device_fingerprint=data.get('device_fingerprint'),
            mac_id=data.get('mac_id'),
            ip_address=ip_address,
            location=str(data.get('location')) if data.get('location') else None
        )
        new_otp.save()

        # Send OTP email
        from app.utils.email_helper import send_otp_email
        email_sent, email_status = send_otp_email(
            to_email=user.email,
            full_name=user.full_name,
            otp_code=otp_code,
            otp_validity_minutes=otp_validity_minutes
        )

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='forgot_password',
            entity_type='otp',
            entity_id=new_otp.otp_id,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={
                "email": email,
                "device_type": data.get('device_type'),
                "location": data.get('location')
            },
            response_status='success',
            additional_info={
                'email_sent': email_sent,
                'email_status': email_status
            }
        ).save()

        # Prepare response data
        response_data = {
            'otp_sent_to': user.email,
            'otp_validity_minutes': otp_validity_minutes
        }

        return success_response(
            message="OTP sent successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            action_type='forgot_password',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/auth/verify-otp', methods=['POST'])
def verify_otp():
    """
    Verify OTP for Password Reset
    ---
    tags:
      - Authentication
    parameters:
      - name: body
        in: body
        required: true
        description: OTP verification data
        schema:
          type: object
          required:
            - email
            - otp
          properties:
            email:
              type: string
              example: "user@example.com"
              description: User's email address
            otp:
              type: string
              example: "123456"
              description: 6-digit OTP code received via email
            device_type:
              type: string
              example: "mobile"
              description: Device type (desktop, mobile, tablet)
            device_fingerprint:
              type: string
              example: "def456ghi789"
              description: Unique device fingerprint
            mac_id:
              type: string
              example: "AA:BB:CC:DD:EE:FF"
              description: MAC ID or IP address
            location:
              type: object
              properties:
                lat:
                  type: number
                  example: 28.6139
                  description: Latitude
                lng:
                  type: number
                  example: 77.2090
                  description: Longitude
      - name: X-Real-IP
        in: header
        type: string
        required: false
        description: Client IP address
      - name: User-Agent
        in: header
        type: string
        required: false
        description: Client user agent
    responses:
      200:
        description: OTP verified successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "OTP verified successfully"
            data:
              type: object
              properties:
                reset_token:
                  type: string
                  example: "unique_reset_token"
                  description: Token to be used for password reset
                token_validity_minutes:
                  type: integer
                  example: 15
                  description: Reset token validity in minutes
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Invalid OTP
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Invalid OTP"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "INVALID_OTP"
                details:
                  type: string
                  example: "OTP is incorrect or expired"
            meta:
              type: object
      404:
        description: Email not found
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 404
            message:
              type: string
              example: "Email not found"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "EMAIL_NOT_FOUND"
                details:
                  type: string
            meta:
              type: object
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get request data
        data = request.get_json()

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Validate required fields
        email = data.get('email', '').strip().lower()
        otp_code = data.get('otp', '').strip()

        if not email or not otp_code:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='verify_otp',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'Email and OTP are required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Email and OTP are required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by email (check if exists first)
        user = User.objects(email=email).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='verify_otp',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Email not found",
                error_code="EMAIL_NOT_FOUND",
                error_details="No user found with this email",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='verify_otp',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'User account is deactivated'}
            ).save()

            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated. Please contact administrator.",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Import OTP and ResetToken models
        from app.models.otp import OTP
        from app.models.reset_token import ResetToken

        # Find valid OTP
        otp_record = OTP.objects(
            user_id=user.user_id,
            email=email,
            otp_code=otp_code,
            otp_type='forgot_password',
            is_used=False,
            is_expired=False
        ).first()

        if not otp_record:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='verify_otp',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'OTP not found or already used'}
            ).save()

            return error_response(
                message="Invalid OTP",
                error_code="INVALID_OTP",
                error_details="OTP is incorrect or expired",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if OTP is valid
        if not otp_record.is_valid():
            # Increment attempts
            otp_record.increment_attempts()

            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='verify_otp',
                entity_type='otp',
                entity_id=otp_record.otp_id,
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"email": email},
                response_status='failed',
                error_details={'reason': 'OTP expired or max attempts reached', 'attempts': otp_record.attempts}
            ).save()

            return error_response(
                message="Invalid OTP",
                error_code="INVALID_OTP",
                error_details="OTP is incorrect or expired",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Mark OTP as used
        otp_record.mark_as_used()

        # Expire any existing reset tokens for this user
        existing_tokens = ResetToken.objects(user_id=user.user_id, is_used=False, is_expired=False)
        for token in existing_tokens:
            token.is_expired = True
            token.save()

        # Generate reset token
        reset_token = ResetToken.generate_reset_token()
        token_validity_minutes = 15

        # Create reset token record
        new_reset_token = ResetToken(
            user_id=user.user_id,
            email=user.email,
            reset_token=reset_token,
            expires_at=datetime.utcnow() + timedelta(minutes=token_validity_minutes),
            device_type=data.get('device_type'),
            device_fingerprint=data.get('device_fingerprint'),
            mac_id=data.get('mac_id'),
            ip_address=ip_address,
            location=str(data.get('location')) if data.get('location') else None,
            otp_id=otp_record.otp_id
        )
        new_reset_token.save()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='verify_otp',
            entity_type='reset_token',
            entity_id=new_reset_token.token_id,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={
                "email": email,
                "device_type": data.get('device_type'),
                "location": data.get('location')
            },
            response_status='success',
            additional_info={
                'otp_id': otp_record.otp_id,
                'token_validity_minutes': token_validity_minutes
            }
        ).save()

        # Prepare response data
        response_data = {
            'reset_token': reset_token,
            'token_validity_minutes': token_validity_minutes
        }

        return success_response(
            message="OTP verified successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            action_type='verify_otp',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/auth/reset-password', methods=['POST'])
def reset_password():
    """
    Reset Password with Token
    ---
    tags:
      - Authentication
    parameters:
      - name: body
        in: body
        required: true
        description: Password reset data
        schema:
          type: object
          required:
            - reset_token
            - new_password
          properties:
            reset_token:
              type: string
              example: "unique_reset_token"
              description: Reset token received after OTP verification
            new_password:
              type: string
              example: "newSecurePassword123"
              description: New password (min 8 characters recommended)
            device_type:
              type: string
              example: "desktop"
              description: Device type (desktop, mobile, tablet)
            device_fingerprint:
              type: string
              example: "jkl012mno345"
              description: Unique device fingerprint
            mac_id:
              type: string
              example: "192.168.1.101"
              description: MAC ID or IP address
            location:
              type: object
              properties:
                lat:
                  type: number
                  example: 28.6139
                  description: Latitude
                lng:
                  type: number
                  example: 77.2090
                  description: Longitude
      - name: X-Real-IP
        in: header
        type: string
        required: false
        description: Client IP address
      - name: User-Agent
        in: header
        type: string
        required: false
        description: Client user agent
    responses:
      200:
        description: Password reset successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "Password reset successfully"
            data:
              type: object
              properties:
                password_reset:
                  type: boolean
                  example: true
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Invalid or expired token
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Invalid or expired token"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "INVALID_RESET_TOKEN"
                details:
                  type: string
                  example: "Reset token is invalid or has expired"
            meta:
              type: object
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get request data
        data = request.get_json()

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Validate required fields
        reset_token = data.get('reset_token', '').strip()
        new_password = data.get('new_password', '').strip()

        if not reset_token or not new_password:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='reset_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'Reset token and new password are required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Reset token and new password are required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Validate password length
        if len(new_password) < 6:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='reset_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'Password too short'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Password must be at least 6 characters long",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Import ResetToken model
        from app.models.reset_token import ResetToken

        # Find valid reset token
        token_record = ResetToken.objects(
            reset_token=reset_token,
            is_used=False,
            is_expired=False
        ).first()

        if not token_record:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='reset_password',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'Reset token not found or already used'}
            ).save()

            return error_response(
                message="Invalid or expired token",
                error_code="INVALID_RESET_TOKEN",
                error_details="Reset token is invalid or has expired",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if token is valid
        if not token_record.is_valid():
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='reset_password',
                entity_type='reset_token',
                entity_id=token_record.token_id,
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'Reset token expired'}
            ).save()

            return error_response(
                message="Invalid or expired token",
                error_code="INVALID_RESET_TOKEN",
                error_details="Reset token is invalid or has expired",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user (check if exists first)
        user = User.objects(user_id=token_record.user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='reset_password',
                entity_type='reset_token',
                entity_id=token_record.token_id,
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Invalid or expired token",
                error_code="INVALID_RESET_TOKEN",
                error_details="Reset token is invalid or has expired",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='reset_password',
                entity_type='reset_token',
                entity_id=token_record.token_id,
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'User account is deactivated'}
            ).save()

            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated. Please contact administrator.",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Update user password
        user.password = new_password
        user.updated_at = datetime.utcnow()
        user.save()

        # Mark token as used
        token_record.mark_as_used()

        # Send password reset success email
        from app.utils.email_helper import send_password_reset_success_email
        email_sent, email_status = send_password_reset_success_email(
            to_email=user.email,
            full_name=user.full_name
        )

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='reset_password',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={
                "device_type": data.get('device_type'),
                "location": data.get('location')
            },
            response_status='success',
            additional_info={
                'reset_token_id': token_record.token_id,
                'email_sent': email_sent,
                'email_status': email_status
            }
        ).save()

        # Prepare response data
        response_data = {
            'password_reset': True
        }

        return success_response(
            message="Password reset successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            action_type='reset_password',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )

@api_v1_bp.route('/auth/refresh-token', methods=['POST'])
def refresh_token():
    """
    Refresh Token Endpoint
    ---
    tags:
      - Authentication
    parameters:
      - name: body
        in: body
        required: true
        description: Refresh token and device information
        schema:
          type: object
          required:
            - refresh_token
          properties:
            refresh_token:
              type: string
              example: "valid_refresh_token_string"
              description: "Valid refresh token from login"
            device_type:
              type: string
              example: "mobile"
              description: "Device type (mobile, web, desktop, etc.)"
            device_fingerprint:
              type: string
              example: "pqr678stu901"
              description: "Unique device fingerprint"
            mac_id:
              type: string
              example: "11:22:33:44:55:66"
              description: "MAC address of device"
            location:
              type: object
              properties:
                lat:
                  type: number
                  example: 28.6139
                lng:
                  type: number
                  example: 77.2090
      - name: X-Real-IP
        in: header
        type: string
        required: false
        description: Client IP address
      - name: User-Agent
        in: header
        type: string
        required: false
        description: Client user agent
    responses:
      200:
        description: Token refreshed successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "Token refreshed successfully"
            data:
              type: object
              properties:
                access_token:
                  type: string
                  description: "New JWT access token"
                refresh_token:
                  type: string
                  description: "New refresh token"
                expires_in:
                  type: integer
                  example: 3600
                  description: "Access token expiry in seconds"
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Refresh token is required"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
            meta:
              type: object
      401:
        description: Invalid or expired refresh token
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 401
            message:
              type: string
              example: "Invalid refresh token"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "INVALID_REFRESH_TOKEN"
                details:
                  type: string
                  example: "Refresh token is invalid or expired"
            meta:
              type: object
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get request data
        data = request.get_json()

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Validate required fields
        refresh_token_str = data.get('refresh_token')

        if not refresh_token_str:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='refresh_token',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'Missing refresh token'}
            ).save()

            return error_response(
                message="Refresh token is required",
                error_code="VALIDATION_ERROR",
                error_details="Missing refresh token in request body",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find refresh token in database
        token_record = RefreshToken.objects(refresh_token=refresh_token_str).first()

        if not token_record:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='refresh_token',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={'token_provided': 'hidden'},
                response_status='failed',
                error_details={'reason': 'Refresh token not found'}
            ).save()

            return error_response(
                message="Invalid refresh token",
                error_code="INVALID_REFRESH_TOKEN",
                error_details="Refresh token is invalid or expired",
                status_code=401,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Validate token
        if not token_record.is_valid():
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=token_record.user_id,
                action_type='refresh_token',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={'token_provided': 'hidden'},
                response_status='failed',
                error_details={
                    'reason': 'Token expired or invalid',
                    'is_used': token_record.is_used,
                    'is_expired': token_record.is_expired,
                    'is_revoked': token_record.is_revoked
                }
            ).save()

            return error_response(
                message="Invalid refresh token",
                error_code="INVALID_REFRESH_TOKEN",
                error_details="Refresh token is invalid or expired",
                status_code=401,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get user (check if exists first)
        user = User.objects(user_id=token_record.user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=token_record.user_id,
                action_type='refresh_token',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Invalid refresh token",
                error_code="INVALID_REFRESH_TOKEN",
                error_details="User associated with token not found",
                status_code=401,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='refresh_token',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'User account is deactivated'}
            ).save()

            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated. Please contact administrator.",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Mark old token as used
        token_record.mark_as_used()

        # Generate new tokens
        new_access_token = create_access_token(identity=user.user_id)
        new_refresh_token_str = create_refresh_token(identity=user.user_id)

        # Create new refresh token record
        import json
        location_data = data.get('location')
        new_token_record = RefreshToken.create_refresh_token(
            user_id=user.user_id,
            device_type=data.get('device_type'),
            device_fingerprint=data.get('device_fingerprint'),
            mac_id=data.get('mac_id'),
            location=json.dumps(location_data) if location_data else None,
            user_agent=user_agent,
            ip_address=ip_address,
            expires_in_days=30  # 30 days validity
        )

        # Update the refresh token string to match JWT refresh token
        new_token_record.refresh_token = new_refresh_token_str
        new_token_record.save()

        # Update session log (create new session or update existing)
        session_log_id = str(uuid.uuid4())
        expires_at = datetime.utcnow() + timedelta(seconds=3600)

        session_log = SessionLog(
            session_log_id=session_log_id,
            user_id=user.user_id,
            access_token=new_access_token,
            refresh_token=new_refresh_token_str,
            device_type=data.get('device_type'),
            device_fingerprint=data.get('device_fingerprint'),
            mac_id=data.get('mac_id'),
            ip_address=ip_address,
            user_agent=user_agent,
            location=data.get('location'),
            is_active=True,
            expires_at=expires_at
        )
        session_log.save()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='refresh_token',
            entity_type='session',
            entity_id=session_log_id,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={
                'device_type': data.get('device_type'),
                'device_fingerprint': data.get('device_fingerprint')
            },
            response_status='success'
        ).save()

        # Prepare response data
        response_data = {
            "access_token": new_access_token,
            "refresh_token": new_refresh_token_str,
            "expires_in": 3600  # Access token expires in 1 hour
        }

        return success_response(
            message="Token refreshed successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            action_type='refresh_token',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )

@api_v1_bp.route('/auth/logout', methods=['POST'])
@jwt_required()
def logout():
    """
    Logout Endpoint
    ---
    tags:
      - Authentication
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT access token)
        example: "Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."
      - name: body
        in: body
        required: false
        description: Logout information with device details
        schema:
          type: object
          properties:
            session_log_id:
              type: string
              example: "uuid"
              description: "Session log ID to terminate (optional)"
            device_type:
              type: string
              example: "desktop"
              description: "Device type (mobile, web, desktop, etc.)"
            device_fingerprint:
              type: string
              example: "vwx234yza567"
              description: "Unique device fingerprint"
            mac_id:
              type: string
              example: "192.168.1.102"
              description: "MAC address or device identifier"
            location:
              type: object
              properties:
                lat:
                  type: number
                  example: 28.6139
                lng:
                  type: number
                  example: 77.2090
      - name: X-Real-IP
        in: header
        type: string
        required: false
        description: Client IP address
      - name: User-Agent
        in: header
        type: string
        required: false
        description: Client user agent
    responses:
      200:
        description: Logged out successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "Logged out successfully"
            data:
              type: object
              properties:
                session_terminated:
                  type: boolean
                  example: true
                session_duration_seconds:
                  type: integer
                  example: 3600
                  description: "Duration of the session in seconds"
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      401:
        description: Unauthorized - Invalid or missing token
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 401
            message:
              type: string
              example: "Unauthorized"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "AUTH_REQUIRED"
                details:
                  type: string
                  example: "Valid authentication token required"
            meta:
              type: object
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current user from JWT token
        current_user_id = get_jwt_identity()

        # Get request data
        data = request.get_json() or {}

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Get the authorization header to extract the access token
        auth_header = request.headers.get('Authorization', '')
        access_token = None
        if auth_header.startswith('Bearer '):
            access_token = auth_header[7:]  # Remove 'Bearer ' prefix

        # Find user
        user = User.objects(user_id=current_user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                action_type='logout',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Unauthorized",
                error_code="AUTH_REQUIRED",
                error_details="User not found",
                status_code=401,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user.user_id,
                action_type='logout',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={},
                response_status='failed',
                error_details={'reason': 'User account is deactivated'}
            ).save()

            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated. Please contact administrator.",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get session_log_id from request or find by access token
        session_log_id = data.get('session_log_id')
        
        if session_log_id:
            # Find specific session by ID
            session = SessionLog.objects(
                session_log_id=session_log_id,
                user_id=current_user_id,
                is_active=True
            ).first()
        elif access_token:
            # Find session by access token
            session = SessionLog.objects(
                access_token=access_token,
                user_id=current_user_id,
                is_active=True
            ).first()
        else:
            # Find any active session for this user
            session = SessionLog.objects(
                user_id=current_user_id,
                is_active=True
            ).first()

        session_duration_seconds = 0
        session_terminated = False

        if session:
            # Calculate session duration
            logout_time = datetime.utcnow()
            login_time = session.login_at or session.created_at
            session_duration = logout_time - login_time
            session_duration_seconds = int(session_duration.total_seconds())

            # Update session log
            session.is_active = False
            session.logout_at = logout_time
            session.updated_at = logout_time
            session.save()

            session_terminated = True

            # Also revoke all refresh tokens for this user (optional - for complete logout)
            # Uncomment if you want to invalidate all refresh tokens on logout
            # RefreshToken.revoke_all_user_tokens(current_user_id)
            
            # Or just revoke the specific refresh token for this session
            if session.refresh_token:
                refresh_token_record = RefreshToken.objects(
                    refresh_token=session.refresh_token,
                    user_id=current_user_id
                ).first()
                if refresh_token_record and not refresh_token_record.is_revoked:
                    refresh_token_record.revoke()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=current_user_id,
            action_type='logout',
            entity_type='session',
            entity_id=session.session_log_id if session else None,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={
                'device_type': data.get('device_type'),
                'device_fingerprint': data.get('device_fingerprint'),
                'session_log_id': session_log_id
            },
            response_status='success'
        ).save()

        # Prepare response data
        response_data = {
            "session_terminated": session_terminated,
            "session_duration_seconds": session_duration_seconds
        }

        return success_response(
            message="Logged out successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            action_type='logout',
            user_id=current_user_id if 'current_user_id' in locals() else None,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )
