from flask import request
from flask_jwt_extended import get_jwt_identity
from app.api.v1 import api_v1_bp
from app.models.user import User
from app.models.action_log import ActionLog
from app.utils.decorators import admin_required
from app.utils.response_helper import success_response, error_response
from datetime import datetime, timedelta
import uuid


@api_v1_bp.route('/admin/users', methods=['GET'])
@admin_required
def admin_get_users():
    """
    Get All Users (Admin Only)
    ---
    tags:
      - Admin
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
        default: Bearer <your_token_here>
      - name: page
        in: query
        type: integer
        required: false
        default: 1
        description: Page number for pagination
      - name: limit
        in: query
        type: integer
        required: false
        default: 50
        description: Number of users per page
      - name: search
        in: query
        type: string
        required: false
        description: Search by name or email
      - name: status
        in: query
        type: string
        required: false
        enum: [active, inactive]
        description: Filter by user status
      - name: user_type
        in: query
        type: string
        required: false
        enum: [US, AD]
        description: Filter by user type
    responses:
      200:
        description: Users fetched successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "Users fetched"
            data:
              type: object
              properties:
                users:
                  type: array
                  items:
                    type: object
                    properties:
                      user_id:
                        type: string
                      email:
                        type: string
                      full_name:
                        type: string
                      role:
                        type: string
                      status:
                        type: string
                      phone:
                        type: string
                      created_at:
                        type: string
                      updated_at:
                        type: string
                      created_by:
                        type: string
                      updated_by:
                        type: string
                pagination:
                  type: object
                  properties:
                    page:
                      type: integer
                    limit:
                      type: integer
                    total:
                      type: integer
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      403:
        description: Access denied
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 403
            message:
              type: string
              example: "Access denied"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "FORBIDDEN"
                details:
                  type: string
                  example: "Admin access required"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current admin user
        admin_user_id = get_jwt_identity()

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 50))
        search = request.args.get('search', '').strip()
        status = request.args.get('status', '').strip()
        user_type_filter = request.args.get('user_type', '').strip()

        # Build query
        query = {}

        # Search filter
        if search:
            query['$or'] = [
                {'email': {'$regex': search, '$options': 'i'}},
                {'full_name': {'$regex': search, '$options': 'i'}}
            ]

        # Status filter
        if status == 'active':
            query['is_active'] = True
        elif status == 'inactive':
            query['is_active'] = False

        # User type filter
        if user_type_filter in ['US', 'AD']:
            query['user_type'] = user_type_filter

        # Calculate pagination
        skip = (page - 1) * limit

        # Get total count
        total_count = User.objects(__raw__=query).count()

        # Fetch users
        users = User.objects(__raw__=query).skip(skip).limit(limit).order_by('-created_at')

        # Prepare user data
        users_data = []
        for user in users:
            users_data.append({
                'user_id': user.user_id,
                'email': user.email,
                'full_name': user.full_name,
                'role': user.user_type,
                'status': 'active' if user.is_active else 'inactive',
                'phone': user.phone,
                'created_at': user.created_at.isoformat() + 'Z' if user.created_at else None,
                'updated_at': user.updated_at.isoformat() + 'Z' if user.updated_at else None,
                'created_by': user.created_by,
                'updated_by': user.updated_by
            })

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_users',
            entity_type='user',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={
                'page': page,
                'limit': limit,
                'search': search,
                'status': status
            },
            response_status='success'
        ).save()

        # Prepare response
        response_data = {
            'users': users_data,
            'pagination': {
                'page': page,
                'limit': limit,
                'total': total_count
            }
        }

        return success_response(
            message="Users fetched",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_users',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/update_users_by_id', methods=['PUT'])
@admin_required
def admin_update_user_by_id():
    """
    Update User By ID (Admin Only)
    ---
    tags:
      - Admin
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
        default: Bearer <your_token_here>
      - name: body
        in: body
        required: true
        description: User update data
        schema:
          type: object
          required:
            - user_id
          properties:
            user_id:
              type: string
              example: "550e8400-e29b-41d4-a716-446655440000"
              description: User ID to update
            full_name:
              type: string
              example: "Jane Smith Updated"
              description: Full name of the user
            email:
              type: string
              example: "jane.updated@example.com"
              description: Email address of the user
            role:
              type: string
              enum: ["US", "AD"]
              example: "US"
              description: User role - US (User) or AD (Admin)
            phone:
              type: string
              example: "+1234567890"
              description: Phone number
            status:
              type: string
              enum: ["active", "inactive"]
              example: "active"
              description: User status
            is_verified:
              type: boolean
              example: true
              description: Email verification status
            password:
              type: string
              example: "NewPassword123"
              description: New password (optional)
    responses:
      200:
        description: User updated successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "User updated"
            data:
              type: object
              properties:
                user_id:
                  type: string
                email:
                  type: string
                full_name:
                  type: string
                role:
                  type: string
                status:
                  type: string
                  example: "active"
                phone:
                  type: string
                is_verified:
                  type: boolean
                updated_at:
                  type: string
                  format: date-time
                updated_by:
                  type: string
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Validation error"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
                  example: "Invalid input data"
                field_errors:
                  type: object
                  example: {"email": "Email already exists"}
            meta:
              type: object
      404:
        description: User not found
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 404
            message:
              type: string
              example: "User not found"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "NOT_FOUND"
                details:
                  type: string
                  example: "User with the specified ID does not exist"
            meta:
              type: object
      403:
        description: Access denied
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 403
            message:
              type: string
              example: "Access denied"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current admin user
        admin_user_id = get_jwt_identity()

        # Get request data
        data = request.get_json()

        # Validate required fields
        user_id = data.get('user_id', '').strip()

        if not user_id:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_update_user',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'user_id is required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="user_id is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by user_id
        user = User.objects(user_id=user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_update_user',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="User not found",
                error_code="NOT_FOUND",
                error_details="User with the specified ID does not exist",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Field validation
        field_errors = {}

        # Get optional fields
        full_name = data.get('full_name', '').strip()
        email = data.get('email', '').strip()
        role = data.get('role', '').strip()
        phone = data.get('phone', '').strip()
        status = data.get('status', '').strip()
        is_verified = data.get('is_verified')
        password = data.get('password', '').strip()

        # Validate email format if provided
        if email and '@' not in email:
            field_errors['email'] = 'Invalid email format'

        # Check if email already exists (for different user)
        if email and email != user.email:
            existing_user = User.objects(email=email).first()
            if existing_user:
                field_errors['email'] = 'Email already exists'

        # Validate role if provided
        if role and role not in ['US', 'AD']:
            field_errors['role'] = 'Role must be either US (User) or AD (Admin)'

        # Validate status if provided
        if status and status not in ['active', 'inactive']:
            field_errors['status'] = 'Status must be either active or inactive'

        if field_errors:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_update_user',
                entity_type='user',
                entity_id=user_id,
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'field_errors': field_errors}
            ).save()

            error_resp = {
                "status": "failed",
                "status_code": 400,
                "message": "Validation error",
                "error": {
                    "code": "VALIDATION_ERROR",
                    "details": "Invalid input data",
                    "field_errors": field_errors
                },
                "meta": {
                    "request_id": request_id,
                    "action_log_id": action_log_id,
                    "timestamp": datetime.utcnow().isoformat() + "Z"
                }
            }
            return error_resp, 400

        # Store old values for logging
        old_values = {
            'email': user.email,
            'full_name': user.full_name,
            'role': user.user_type,
            'status': 'active' if user.is_active else 'inactive'
        }

        # Update user fields
        if full_name:
            user.full_name = full_name
        if email:
            user.email = email
        if role:
            user.user_type = role
        if phone:
            user.phone = phone
        if status:
            user.is_active = (status == 'active')
        if is_verified is not None:
            user.is_verified = is_verified
        if password:
            user.password = password

        # Update audit fields
        user.updated_by = admin_user_id
        user.updated_at = datetime.utcnow()

        # Save user
        user.save()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_update_user',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success',
            additional_info={
                'old_values': old_values,
                'updated_fields': list(data.keys())
            }
        ).save()

        # Prepare response data
        response_data = {
            'user_id': user.user_id,
            'email': user.email,
            'full_name': user.full_name,
            'role': user.user_type,
            'status': 'active' if user.is_active else 'inactive',
            'phone': user.phone,
            'is_verified': user.is_verified,
            'updated_at': user.updated_at.isoformat() + 'Z' if user.updated_at else None,
            'updated_by': user.updated_by
        }

        return success_response(
            message="User updated",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_update_user',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/create_users', methods=['POST'])
@admin_required
def admin_create_user():
    """
    Create New User (Admin Only)
    ---
    tags:
      - Admin
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
        default: Bearer <your_token_here>
      - name: body
        in: body
        required: true
        description: User creation data
        schema:
          type: object
          required:
            - full_name
            - email
            - role
          properties:
            full_name:
              type: string
              example: "Jane Smith"
              description: Full name of the user
            email:
              type: string
              example: "jane@example.com"
              description: Email address of the user
            role:
              type: string
              enum: ["US", "AD"]
              example: "US"
              description: User role - US (User) or AD (Admin)
            phone:
              type: string
              example: "+1234567890"
              description: Phone number (optional)
    responses:
      201:
        description: User created successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 201
            message:
              type: string
              example: "User created"
            data:
              type: object
              properties:
                user_id:
                  type: string
                email:
                  type: string
                full_name:
                  type: string
                role:
                  type: string
                temp_password:
                  type: string
                  description: Temporary password for the user
                status:
                  type: string
                  example: "active"
                phone:
                  type: string
                created_at:
                  type: string
                  format: date-time
                created_by:
                  type: string
                updated_at:
                  type: string
                  format: date-time
                updated_by:
                  type: string
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Validation error"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
                  example: "Invalid input data"
                field_errors:
                  type: object
                  example: {"email": "Email already exists"}
            meta:
              type: object
      403:
        description: Access denied
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 403
            message:
              type: string
              example: "Access denied"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current admin user
        admin_user_id = get_jwt_identity()

        # Get request data
        data = request.get_json()

        # Validate required fields
        full_name = data.get('full_name', '').strip()
        email = data.get('email', '').strip()
        role = data.get('role', '').strip()
        phone = data.get('phone', '').strip()

        # Field validation
        field_errors = {}

        if not full_name:
            field_errors['full_name'] = 'Full name is required'

        if not email:
            field_errors['email'] = 'Email is required'
        elif '@' not in email:
            field_errors['email'] = 'Invalid email format'

        if not role:
            field_errors['role'] = 'Role is required'
        elif role not in ['US', 'AD']:
            field_errors['role'] = 'Role must be either US (User) or AD (Admin)'

        # Check if email already exists
        if email and User.objects(email=email).first():
            field_errors['email'] = 'Email already exists'

        if field_errors:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_create_user',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'field_errors': field_errors}
            ).save()

            error_resp = {
                "status": "failed",
                "status_code": 400,
                "message": "Validation error",
                "error": {
                    "code": "VALIDATION_ERROR",
                    "details": "Invalid input data",
                    "field_errors": field_errors
                },
                "meta": {
                    "request_id": request_id,
                    "action_log_id": action_log_id,
                    "timestamp": datetime.utcnow().isoformat() + "Z"
                }
            }
            return error_resp, 400

        # Generate temporary password
        import random
        import string
        temp_password = ''.join(random.choices(string.ascii_letters + string.digits, k=12))

        # Create new user
        new_user = User(
            email=email,
            password=temp_password,  # Store plain password as per project requirement
            full_name=full_name,
            user_type=role,
            phone=phone if phone else None,
            is_active=True,
            is_verified=False,
            created_by=admin_user_id,
            updated_by=admin_user_id,
            created_at=datetime.utcnow(),
            updated_at=datetime.utcnow()
        )
        new_user.save()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_create_user',
            entity_type='user',
            entity_id=new_user.user_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={
                'email': email,
                'full_name': full_name,
                'role': role
            },
            response_status='success'
        ).save()

        # Prepare response data
        response_data = {
            'user_id': new_user.user_id,
            'email': new_user.email,
            'full_name': new_user.full_name,
            'role': new_user.user_type,
            'temp_password': temp_password,
            'status': 'active',
            'phone': new_user.phone,
            'created_at': new_user.created_at.isoformat() + 'Z' if new_user.created_at else None,
            'created_by': new_user.created_by,
            'updated_at': new_user.updated_at.isoformat() + 'Z' if new_user.updated_at else None,
            'updated_by': new_user.updated_by
        }

        return success_response(
            message="User created",
            data=response_data,
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_create_user',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/get_users_by_id', methods=['GET'])
@admin_required
def admin_get_user_by_id():
    """
    Get User By ID (Admin Only)
    ---
    tags:
      - Admin
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
        default: Bearer <your_token_here>
      - name: user_id
        in: query
        type: string
        required: true
        description: User ID to fetch
        example: "550e8400-e29b-41d4-a716-446655440000"
    responses:
      200:
        description: User fetched successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "User fetched"
            data:
              type: object
              properties:
                user_id:
                  type: string
                email:
                  type: string
                full_name:
                  type: string
                role:
                  type: string
                status:
                  type: string
                  example: "active"
                phone:
                  type: string
                is_verified:
                  type: boolean
                profile_data:
                  type: object
                created_at:
                  type: string
                  format: date-time
                created_by:
                  type: string
                updated_at:
                  type: string
                  format: date-time
                updated_by:
                  type: string
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Validation error"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
                  example: "user_id is required"
            meta:
              type: object
      404:
        description: User not found
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 404
            message:
              type: string
              example: "User not found"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "NOT_FOUND"
                details:
                  type: string
                  example: "User with the specified ID does not exist"
            meta:
              type: object
      403:
        description: Access denied
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 403
            message:
              type: string
              example: "Access denied"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current admin user
        admin_user_id = get_jwt_identity()

        # Get user_id from query parameters
        user_id = request.args.get('user_id', '').strip()

        # Validate user_id
        if not user_id:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_get_user_by_id',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'user_id': user_id},
                response_status='failed',
                error_details={'reason': 'user_id is required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="user_id is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by user_id
        user = User.objects(user_id=user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_get_user_by_id',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'user_id': user_id},
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="User not found",
                error_code="NOT_FOUND",
                error_details="User with the specified ID does not exist",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_user_by_id',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'user_id': user_id},
            response_status='success'
        ).save()

        # Prepare user data
        user_data = {
            'user_id': user.user_id,
            'email': user.email,
            'full_name': user.full_name,
            'role': user.user_type,
            'status': 'active' if user.is_active else 'inactive',
            'phone': user.phone,
            'is_verified': user.is_verified,
            'profile_data': user.profile_data if user.profile_data else {},
            'created_at': user.created_at.isoformat() + 'Z' if user.created_at else None,
            'created_by': user.created_by,
            'updated_at': user.updated_at.isoformat() + 'Z' if user.updated_at else None,
            'updated_by': user.updated_by
        }

        return success_response(
            message="User fetched",
            data=user_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_user_by_id',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'user_id': user_id if 'user_id' in locals() else ''},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/delete_users', methods=['DELETE'])
@admin_required
def admin_delete_user():
    """
    Delete User (Admin Only)
    ---
    tags:
      - Admin
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
        default: Bearer <your_token_here>
      - name: body
        in: body
        required: true
        description: User deletion data
        schema:
          type: object
          required:
            - user_id
          properties:
            user_id:
              type: string
              example: "550e8400-e29b-41d4-a716-446655440000"
              description: User ID to delete
    responses:
      200:
        description: User deleted successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "User deleted"
            data:
              type: object
              properties:
                user_id:
                  type: string
                email:
                  type: string
                full_name:
                  type: string
                deleted_at:
                  type: string
                  format: date-time
                deleted_by:
                  type: string
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Validation error"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
                  example: "user_id is required"
            meta:
              type: object
      404:
        description: User not found
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 404
            message:
              type: string
              example: "User not found"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "NOT_FOUND"
                details:
                  type: string
                  example: "User with the specified ID does not exist"
            meta:
              type: object
      403:
        description: Access denied
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 403
            message:
              type: string
              example: "Access denied"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current admin user
        admin_user_id = get_jwt_identity()

        # Get request data
        data = request.get_json()

        # Validate required fields
        user_id = data.get('user_id', '').strip()

        if not user_id:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_delete_user',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'user_id is required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="user_id is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by user_id
        user = User.objects(user_id=user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_delete_user',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="User not found",
                error_code="NOT_FOUND",
                error_details="User with the specified ID does not exist",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Store user data before deletion
        deleted_user_data = {
            'user_id': user.user_id,
            'email': user.email,
            'full_name': user.full_name,
            'role': user.user_type
        }

        # Delete the user
        user.delete()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_delete_user',
            entity_type='user',
            entity_id=user_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success',
            additional_info={
                'deleted_user': deleted_user_data
            }
        ).save()

        # Prepare response data
        response_data = {
            'user_id': deleted_user_data['user_id'],
            'email': deleted_user_data['email'],
            'full_name': deleted_user_data['full_name'],
            'deleted_at': datetime.utcnow().isoformat() + 'Z',
            'deleted_by': admin_user_id
        }

        return success_response(
            message="User deleted",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_delete_user',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/reset_user_password_by_id', methods=['PUT'])
@admin_required
def admin_reset_user_password():
    """
    Reset User Password By ID (Admin Only)
    ---
    tags:
      - Admin
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
        default: Bearer <your_token_here>
      - name: body
        in: body
        required: true
        description: Password reset data
        schema:
          type: object
          required:
            - user_id
          properties:
            user_id:
              type: string
              example: "550e8400-e29b-41d4-a716-446655440000"
              description: User ID whose password needs to be reset
            new_password:
              type: string
              example: "NewSecurePass123"
              description: New password (optional - if not provided, a random password will be generated)
            send_email:
              type: boolean
              example: true
              default: true
              description: Whether to send email notification to user (default true)
    responses:
      200:
        description: Password reset successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: "Password reset successfully"
            data:
              type: object
              properties:
                user_id:
                  type: string
                email:
                  type: string
                full_name:
                  type: string
                new_password:
                  type: string
                  description: The new password (only returned in response)
                email_sent:
                  type: boolean
                  description: Whether email notification was sent
                email_status:
                  type: string
                  description: Email sending status message
                reset_at:
                  type: string
                  format: date-time
                reset_by:
                  type: string
            meta:
              type: object
              properties:
                request_id:
                  type: string
                action_log_id:
                  type: string
                timestamp:
                  type: string
      400:
        description: Validation error
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 400
            message:
              type: string
              example: "Validation error"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "VALIDATION_ERROR"
                details:
                  type: string
                  example: "user_id is required"
            meta:
              type: object
      404:
        description: User not found
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 404
            message:
              type: string
              example: "User not found"
            error:
              type: object
              properties:
                code:
                  type: string
                  example: "NOT_FOUND"
                details:
                  type: string
                  example: "User with the specified ID does not exist"
            meta:
              type: object
      403:
        description: Access denied
        schema:
          type: object
          properties:
            status:
              type: string
              example: "failed"
            status_code:
              type: integer
              example: 403
            message:
              type: string
              example: "Access denied"
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        # Get current admin user
        admin_user_id = get_jwt_identity()

        # Get request data
        data = request.get_json()

        # Validate required fields
        user_id = data.get('user_id', '').strip()
        new_password = data.get('new_password', '').strip()
        send_email = data.get('send_email', True)

        if not user_id:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_reset_password',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'user_id is required'}
            ).save()

            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="user_id is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find user by user_id
        user = User.objects(user_id=user_id).first()

        if not user:
            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_reset_password',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="User not found",
                error_code="NOT_FOUND",
                error_details="User with the specified ID does not exist",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Generate new password if not provided
        if not new_password:
            import random
            import string
            new_password = ''.join(random.choices(string.ascii_letters + string.digits + string.punctuation, k=16))

        # Store old password (for logging purposes only - not returned)
        old_password_hash = user.password[:10] + "..." if user.password else "N/A"

        # Update user password
        user.password = new_password
        user.updated_by = admin_user_id
        user.updated_at = datetime.utcnow()
        user.save()

        # Send email notification
        email_sent = False
        email_status = "Email not sent (send_email=false)"

        if send_email:
            from app.utils.email_helper import send_password_reset_email
            email_sent, email_status = send_password_reset_email(
                to_email=user.email,
                full_name=user.full_name,
                new_password=new_password
            )

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_reset_password',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={
                'user_id': user_id,
                'send_email': send_email
            },
            response_status='success',
            additional_info={
                'email_sent': email_sent,
                'email_status': email_status,
                'old_password_hash': old_password_hash
            }
        ).save()

        # Prepare response data
        response_data = {
            'user_id': user.user_id,
            'email': user.email,
            'full_name': user.full_name,
            'new_password': new_password,
            'email_sent': email_sent,
            'email_status': email_status,
            'reset_at': datetime.utcnow().isoformat() + 'Z',
            'reset_by': admin_user_id
        }

        return success_response(
            message="Password reset successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        # Log exception
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_reset_password',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )

# ==================== DOCUMENT MANAGEMENT ====================

@api_v1_bp.route('/admin/update_documents/<document_id>', methods=['PUT'])
@admin_required
def admin_update_document(document_id):
    """
    Update Document (Admin Only)
    ---
    tags:
      - Admin - Documents
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
      - name: document_id
        in: path
        type: string
        required: true
        description: Document ID to update
      - name: body
        in: body
        required: true
        schema:
          type: object
          properties:
            document_type:
              type: string
              enum: [question, answer, evidence, contract, report, other, general, product]
            processing_status:
              type: string
              enum: [queued, processing, completed, failed]
            description:
              type: string
              description: Document description
            file_name:
              type: string
              description: Update document file name
    responses:
      200:
        description: Document updated successfully
      404:
        description: Document not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.document import CaseDocument

        admin_user_id = get_jwt_identity()
        data = request.get_json()

        # Find document
        document = CaseDocument.objects(document_id=document_id, is_deleted=False).first()
        if not document:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_update_document',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'Document not found'}
            ).save()

            return error_response(
                message="Document not found",
                error_code="DOCUMENT_NOT_FOUND",
                error_details="No document found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Update fields
        if 'document_type' in data:
            document.document_type = data['document_type']
        if 'processing_status' in data:
            document.processing_status = data['processing_status']
        if 'description' in data:
            document.description = data['description']
        if 'file_name' in data:
            document.original_file_name = data['file_name']

        document.updated_at = datetime.utcnow()
        document.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_update_document',
            entity_type='document',
            entity_id=document.document_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success'
        ).save()

        return success_response(
            message="Document updated successfully",
            data={
                'document_id': document.document_id,
                'file_name': document.original_file_name,
                'document_type': document.document_type,
                'description': document.description if document.description else '',
                'updated_at': document.updated_at.isoformat() + 'Z'
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_update_document',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ==================== ADMIN DOCUMENT MANAGEMENT ====================

@api_v1_bp.route('/admin/upload_document', methods=['POST'])
@admin_required
def admin_upload_document():
    """
    Upload Document (Admin Only)
    ---
    tags:
      - Admin - Documents
    security:
      - Bearer: []
    consumes:
      - multipart/form-data
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
      - name: file
        in: formData
        type: file
        required: true
        description: Document file to upload
      - name: document_type
        in: formData
        type: string
        required: true
        enum: [general, product]
        description: Type of document
      - name: description
        in: formData
        type: string
        required: false
        description: Document description
      - name: case_id
        in: formData
        type: string
        required: false
        description: Associated case ID (optional)
    responses:
      201:
        description: Document uploaded successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: "success"
            status_code:
              type: integer
              example: 201
            message:
              type: string
              example: "Document uploaded successfully"
            data:
              type: object
              properties:
                document_id:
                  type: string
                file_name:
                  type: string
                document_type:
                  type: string
                uploaded_by:
                  type: string
                uploaded_at:
                  type: string
                  format: date-time
            meta:
              type: object
      400:
        description: Invalid type or file missing
      401:
        description: Unauthorized
      403:
        description: Forbidden
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.document import CaseDocument
        from werkzeug.utils import secure_filename
        import os

        admin_user_id = get_jwt_identity()

        # Get admin user info
        admin_user = User.objects(user_id=admin_user_id).first()
        admin_name = admin_user.full_name if admin_user else "Admin"

        # Check if file is present
        if 'file' not in request.files:
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="File is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        file = request.files['file']
        document_type = request.form.get('document_type', '').strip()
        description = request.form.get('description', '').strip()
        case_id = request.form.get('case_id', '').strip()

        # Validate file
        if file.filename == '':
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="No file selected",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Validate document type
        if not document_type or document_type not in ['general', 'product']:
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="document_type must be 'general' or 'product'",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Secure filename
        original_filename = file.filename
        filename = secure_filename(original_filename)
        file_extension = CaseDocument.get_file_extension(filename)

        # Generate unique filename
        unique_filename = f"{uuid.uuid4()}_{filename}"

        # Create upload directory if it doesn't exist
        upload_dir = os.path.join('uploads', 'admin_documents')
        os.makedirs(upload_dir, exist_ok=True)

        # Save file
        file_path = os.path.join(upload_dir, unique_filename)
        file.save(file_path)

        # Get file size
        file_size = os.path.getsize(file_path)

        # Create document record
        new_document = CaseDocument(
            case_id=case_id if case_id else None,
            user_id=admin_user_id,
            uploaded_by=admin_name,
            file_name=unique_filename,
            original_file_name=original_filename,
            file_path=file_path,
            file_size=file_size,
            file_extension=file_extension,
            document_type=document_type,
            description=description,
            processing_status='completed',
            uploaded_at=datetime.utcnow(),
            created_at=datetime.utcnow(),
            updated_at=datetime.utcnow()
        )
        new_document.save()

        # Update case document count if case_id provided
        if case_id:
            from app.models.case import Case
            case = Case.objects(case_id=case_id).first()
            if case:
                case.documents_count += 1
                case.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_upload_document',
            entity_type='document',
            entity_id=new_document.document_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={
                'document_type': document_type,
                'file_name': original_filename,
                'case_id': case_id
            },
            response_status='success'
        ).save()

        return success_response(
            message="Document uploaded successfully",
            data={
                'document_id': new_document.document_id,
                'file_name': original_filename,
                'document_type': new_document.document_type,
                'uploaded_by': admin_name,
                'uploaded_at': new_document.uploaded_at.isoformat() + 'Z'
            },
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_upload_document',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/get_all_documents', methods=['GET'])
@admin_required
def admin_get_all_documents():
    """
    Get All Documents (Admin Only)
    ---
    tags:
      - Admin - Documents
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
      - name: page
        in: query
        type: integer
        default: 1
        description: Page number
      - name: limit
        in: query
        type: integer
        default: 20
        description: Items per page
      - name: document_type
        in: query
        type: string
        enum: [general, product, all]
        description: Filter by document type
    responses:
      200:
        description: Documents fetched successfully
      401:
        description: Unauthorized
      404:
        description: No documents found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.document import CaseDocument

        admin_user_id = get_jwt_identity()

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 20))
        document_type_filter = request.args.get('document_type', 'all').strip()

        # Build query
        query = {'is_deleted': False}

        if document_type_filter and document_type_filter != 'all':
            query['document_type'] = document_type_filter

        # Pagination
        skip = (page - 1) * limit
        total_count = CaseDocument.objects(__raw__=query).count()

        if total_count == 0:
            return error_response(
                message="No documents found",
                error_code="NOT_FOUND",
                error_details="No documents match the criteria",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Fetch documents
        documents = CaseDocument.objects(__raw__=query).skip(skip).limit(limit).order_by('-uploaded_at')

        # Prepare response data
        documents_data = []
        for doc in documents:
            documents_data.append({
                'document_id': doc.document_id,
                'file_name': doc.original_file_name,
                'document_type': doc.document_type,
                'uploaded_by': doc.uploaded_by if doc.uploaded_by else 'Unknown',
                'uploaded_at': doc.uploaded_at.isoformat() + 'Z' if doc.uploaded_at else None,
                'case_id': doc.case_id if doc.case_id else None,
                'description': doc.description if doc.description else ''
            })

        # Calculate total pages
        total_pages = (total_count + limit - 1) // limit

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_all_documents',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'page': page, 'limit': limit, 'document_type': document_type_filter},
            response_status='success'
        ).save()

        return success_response(
            message="Documents fetched successfully",
            data={
                'documents': documents_data,
                'pagination': {
                    'page': page,
                    'limit': limit,
                    'total_pages': total_pages,
                    'total_items': total_count
                }
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_all_documents',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/delete_documents/<document_id>', methods=['DELETE'])
@admin_required
def admin_delete_document(document_id):
    """
    Delete Document (Admin Only)
    ---
    tags:
      - Admin - Documents
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
      - name: document_id
        in: path
        type: string
        required: true
        description: Document ID to delete
    responses:
      200:
        description: Document deleted successfully
      401:
        description: Unauthorized
      403:
        description: Forbidden
      404:
        description: Document not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.document import CaseDocument

        admin_user_id = get_jwt_identity()

        # Find document
        document = CaseDocument.objects(document_id=document_id, is_deleted=False).first()
        if not document:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_delete_document',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'document_id': document_id},
                response_status='failed',
                error_details={'reason': 'Document not found'}
            ).save()

            return error_response(
                message="Document not found",
                error_code="DOCUMENT_NOT_FOUND",
                error_details="No document found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Soft delete the document
        document.soft_delete(admin_user_id)

        # Update case document count if case_id exists
        if document.case_id:
            from app.models.case import Case
            case = Case.objects(case_id=document.case_id).first()
            if case and case.documents_count > 0:
                case.documents_count -= 1
                case.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_delete_document',
            entity_type='document',
            entity_id=document.document_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'document_id': document_id},
            response_status='success'
        ).save()

        return success_response(
            message="Document deleted successfully",
            data={},
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_delete_document',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'document_id': document_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ==================== CASE MANAGEMENT ====================

@api_v1_bp.route('/admin/get_all_cases', methods=['GET'])
@admin_required
def admin_get_all_cases():
    """
    Get All Cases (Admin Only)
    ---
    tags:
      - Admin - Cases
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: page
        in: query
        type: integer
        default: 1
      - name: limit
        in: query
        type: integer
        default: 50
      - name: client
        in: query
        type: string
        description: Filter by client name
      - name: status
        in: query
        type: string
        enum: [all, active, inactive, closed, pending, archived]
        default: all
    responses:
      200:
        description: Cases fetched successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.case import Case

        admin_user_id = get_jwt_identity()

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 50))
        client_filter = request.args.get('client', '').strip()
        status_filter = request.args.get('status', 'all').strip()

        # Build query
        query = {'is_deleted': False}

        if client_filter:
            query['client'] = {'$regex': client_filter, '$options': 'i'}

        if status_filter and status_filter != 'all':
            query['status'] = status_filter

        # Pagination
        skip = (page - 1) * limit
        total_count = Case.objects(__raw__=query).count()

        # Fetch cases
        cases = Case.objects(__raw__=query).skip(skip).limit(limit).order_by('-created_at')

        # Prepare response data
        cases_data = []
        for case in cases:
            # Get user name
            user_name = "Unknown"
            user = User.objects(user_id=case.user_id).first()
            if user:
                user_name = user.full_name

            cases_data.append({
                'case_id': case.case_id,
                'name': case.name,
                'case_number': case.case_number,
                'client': case.client,
                'user_name': user_name,
                'status': case.status,
                'created_at': case.created_at.isoformat() + 'Z' if case.created_at else None,
                'start_date': case.start_date.isoformat() + 'Z' if case.start_date else None,
                'end_date': case.end_date.isoformat() + 'Z' if case.end_date else None,
                'ai_exports': case.ai_exports_count,
                'documents_count': case.documents_count
            })

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_all_cases',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'page': page, 'limit': limit, 'client': client_filter, 'status': status_filter},
            response_status='success'
        ).save()

        return success_response(
            message="Cases fetched",
            data={
                'cases': cases_data,
                'pagination': {
                    'page': page,
                    'limit': limit,
                    'total': total_count
                }
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_all_cases',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/get_case_by_id/<case_id>', methods=['GET'])
@admin_required
def admin_get_case_by_id(case_id):
    """
    Get Case By ID (Admin Only)
    ---
    tags:
      - Admin - Cases
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: case_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Case details fetched
      404:
        description: Case not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.case import Case
        from app.models.document import CaseDocument

        admin_user_id = get_jwt_identity()

        # Find case
        case = Case.objects(case_id=case_id, is_deleted=False).first()
        if not case:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_get_case_by_id',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'case_id': case_id},
                response_status='failed',
                error_details={'reason': 'Case not found'}
            ).save()

            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="No case found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get user name
        user_name = "Unknown"
        user = User.objects(user_id=case.user_id).first()
        if user:
            user_name = user.full_name

        # Get documents for this case
        documents = CaseDocument.objects(case_id=case_id, is_deleted=False)
        documents_data = [{
            'document_id': doc.document_id,
            'file_name': doc.file_name,
            'document_type': doc.document_type,
            'processing_status': doc.processing_status,
            'created_at': doc.created_at.isoformat() + 'Z' if doc.created_at else None
        } for doc in documents]

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_case_by_id',
            entity_type='case',
            entity_id=case.case_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'case_id': case_id},
            response_status='success'
        ).save()

        return success_response(
            message="Case details fetched",
            data={
                'case_id': case.case_id,
                'name': case.name,
                'case_number': case.case_number,
                'client': case.client,
                'description': case.description,
                'case_type': case.case_type,
                'priority': case.priority,
                'status': case.status,
                'user_id': case.user_id,
                'user_name': user_name,
                'created_at': case.created_at.isoformat() + 'Z' if case.created_at else None,
                'start_date': case.start_date.isoformat() + 'Z' if case.start_date else None,
                'end_date': case.end_date.isoformat() + 'Z' if case.end_date else None,
                'ai_exports': case.ai_exports_count,
                'documents': documents_data,
                'ai_responses': []
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_case_by_id',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'case_id': case_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/update_case_status/<case_id>', methods=['PUT'])
@admin_required
def admin_update_case_status(case_id):
    """
    Update Case Status (Admin Only)
    ---
    tags:
      - Admin - Cases
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: case_id
        in: path
        type: string
        required: true
      - name: body
        in: body
        required: true
        schema:
          type: object
          required:
            - status
          properties:
            status:
              type: string
              enum: [active, inactive, closed, pending, archived]
    responses:
      200:
        description: Case status updated
      404:
        description: Case not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.case import Case

        admin_user_id = get_jwt_identity()
        data = request.get_json()

        # Validate status
        status = data.get('status', '').strip()
        if not status:
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="status is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        if status not in ['active', 'inactive', 'closed', 'pending', 'archived']:
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Invalid status value",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find case
        case = Case.objects(case_id=case_id, is_deleted=False).first()
        if not case:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_update_case_status',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data=data,
                response_status='failed',
                error_details={'reason': 'Case not found'}
            ).save()

            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="No case found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Update status
        old_status = case.status
        case.status = status
        case.updated_at = datetime.utcnow()
        case.updated_by = admin_user_id
        case.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_update_case_status',
            entity_type='case',
            entity_id=case.case_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success',
            additional_info={'old_status': old_status, 'new_status': status}
        ).save()

        return success_response(
            message="Case status updated",
            data={
                'case_id': case.case_id,
                'status': case.status,
                'updated_at': case.updated_at.isoformat() + 'Z'
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_update_case_status',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/delete_case_permanent/<case_id>', methods=['DELETE'])
@admin_required
def admin_delete_case_permanent(case_id):
    """
    Permanently Delete Case (Admin Only)
    ---
    tags:
      - Admin - Cases
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: case_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Case permanently deleted
      404:
        description: Case not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.case import Case
        from app.models.document import CaseDocument

        admin_user_id = get_jwt_identity()

        # Find case
        case = Case.objects(case_id=case_id).first()
        if not case:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_delete_case_permanent',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'case_id': case_id},
                response_status='failed',
                error_details={'reason': 'Case not found'}
            ).save()

            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="No case found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Count documents
        documents_count = CaseDocument.objects(case_id=case_id).count()

        # Delete all associated documents
        CaseDocument.objects(case_id=case_id).delete()

        # Delete case
        case_data = {
            'case_id': case.case_id,
            'name': case.name,
            'case_number': case.case_number,
            'client': case.client
        }
        case.delete()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_delete_case_permanent',
            entity_type='case',
            entity_id=case_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'case_id': case_id},
            response_status='success',
            additional_info={'case_data': case_data, 'documents_deleted': documents_count}
        ).save()

        return success_response(
            message="Case permanently deleted",
            data={
                'case_id': case_id,
                'deleted_at': datetime.utcnow().isoformat() + 'Z',
                'documents_deleted': documents_count,
                'cannot_be_restored': True
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_delete_case_permanent',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'case_id': case_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/get_case_analytics', methods=['GET'])
@admin_required
def admin_get_case_analytics():
    """
    Get Case Analytics (Admin Only)
    ---
    tags:
      - Admin - Cases
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: start_date
        in: query
        type: string
        format: date
        description: Start date for analytics (YYYY-MM-DD)
      - name: end_date
        in: query
        type: string
        format: date
        description: End date for analytics (YYYY-MM-DD)
    responses:
      200:
        description: Analytics fetched
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.case import Case
        from datetime import datetime as dt

        admin_user_id = get_jwt_identity()

        # Get query parameters
        start_date_str = request.args.get('start_date', '').strip()
        end_date_str = request.args.get('end_date', '').strip()

        # Build query
        query = {'is_deleted': False}

        if start_date_str:
            try:
                start_date = dt.strptime(start_date_str, '%Y-%m-%d')
                query['created_at__gte'] = start_date
            except ValueError:
                pass

        if end_date_str:
            try:
                end_date = dt.strptime(end_date_str, '%Y-%m-%d')
                query['created_at__lte'] = end_date
            except ValueError:
                pass

        # Get counts
        total_cases = Case.objects(__raw__=query).count()
        active_cases = Case.objects(__raw__={**query, 'status': 'active'}).count()
        closed_cases = Case.objects(__raw__={**query, 'status': 'closed'}).count()
        archived_cases = Case.objects(__raw__={**query, 'status': 'archived'}).count()

        # Cases by month (simplified - last 12 months)
        cases_by_month = []
        # This is a simplified version - in production you'd use aggregation
        for i in range(12):
            month_str = (dt.utcnow().replace(day=1) - timedelta(days=30*i)).strftime('%Y-%m')
            cases_by_month.insert(0, {'month': month_str, 'count': 0})

        # Cases by type
        cases_by_type = {}
        if 'case_type' in Case._fields:
            contract_cases = Case.objects(__raw__={**query, 'case_type': 'contract'}).count()
            litigation_cases = Case.objects(__raw__={**query, 'case_type': 'litigation'}).count()
            cases_by_type = {
                'contract': contract_cases,
                'litigation': litigation_cases
            }

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_case_analytics',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'start_date': start_date_str, 'end_date': end_date_str},
            response_status='success'
        ).save()

        return success_response(
            message="Analytics fetched",
            data={
                'total_cases': total_cases,
                'active_cases': active_cases,
                'closed_cases': closed_cases,
                'archived_cases': archived_cases,
                'cases_by_month': cases_by_month,
                'cases_by_type': cases_by_type
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_case_analytics',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )
# ==================== CLIENT MANAGEMENT ====================

@api_v1_bp.route('/admin/add_client', methods=['POST'])
@admin_required
def admin_add_client():
    """
    Add New Client (Admin Only)
    ---
    tags:
      - Admin - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: body
        in: body
        required: true
        schema:
          type: object
          required:
            - client_name
          properties:
            client_name:
              type: string
              example: "Stockholm"
    responses:
      201:
        description: Client added successfully
      400:
        description: Validation error
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client

        admin_user_id = get_jwt_identity()
        data = request.get_json()

        # Validate
        client_name = data.get('client_name', '').strip()
        if not client_name:
            return error_response(
                message="Client name is required",
                error_code="VALIDATION_ERROR",
                error_details="client_name is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if client already exists
        existing_client = Client.objects(client_name=client_name, is_deleted=False).first()
        if existing_client:
            return error_response(
                message="Client already exists",
                error_code="VALIDATION_ERROR",
                error_details="Client with this name already exists",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Create client
        new_client = Client(
            client_name=client_name,
            created_by=admin_user_id,
            created_at=datetime.utcnow(),
            updated_at=datetime.utcnow()
        )
        new_client.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_add_client',
            entity_type='client',
            entity_id=new_client.client_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success'
        ).save()

        return success_response(
            message="Client added successfully",
            data={
                'client_id': new_client.client_id,
                'client_name': new_client.client_name,
                'created_at': new_client.created_at.isoformat() + 'Z'
            },
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_add_client',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/get_all_clients', methods=['GET'])
@admin_required
def admin_get_all_clients():
    """
    Get All Clients (Admin Only)
    ---
    tags:
      - Admin - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: page
        in: query
        type: integer
        default: 1
      - name: limit
        in: query
        type: integer
        default: 50
    responses:
      200:
        description: Clients fetched successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client
        from app.models.case import Case

        admin_user_id = get_jwt_identity()

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 50))

        # Pagination
        skip = (page - 1) * limit
        total_count = Client.objects(is_deleted=False).count()

        # Fetch clients
        clients = Client.objects(is_deleted=False).skip(skip).limit(limit).order_by('-created_at')

        # Prepare response data
        clients_data = []
        for client in clients:
            # Update case counts
            client.update_case_counts()

            clients_data.append({
                'client_id': client.client_id,
                'client_name': client.client_name,
                'active_cases': client.active_cases_count,
                'inactive_cases': client.inactive_cases_count,
                'total_cases': client.total_cases_count
            })

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_all_clients',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'page': page, 'limit': limit},
            response_status='success'
        ).save()

        return success_response(
            message="Clients fetched successfully",
            data={
                'clients': clients_data,
                'pagination': {
                    'page': page,
                    'limit': limit,
                    'total': total_count
                }
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_all_clients',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/get_client_by_id/<client_id>', methods=['GET'])
@admin_required
def admin_get_client_by_id(client_id):
    """
    Get Client By ID (Admin Only)
    ---
    tags:
      - Admin - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: client_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Client details fetched
      404:
        description: Client not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client
        from app.models.case import Case

        admin_user_id = get_jwt_identity()

        # Find client
        client = Client.objects(client_id=client_id, is_deleted=False).first()
        if not client:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_get_client_by_id',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'client_id': client_id},
                response_status='failed',
                error_details={'reason': 'Client not found'}
            ).save()

            return error_response(
                message="Client not found",
                error_code="CLIENT_NOT_FOUND",
                error_details="No client found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get cases for this client
        cases = Case.objects(client_id=client_id, is_deleted=False)
        cases_data = [{
            'case_id': case.case_id,
            'name': case.name,
            'case_number': case.case_number,
            'status': case.status,
            'start_date': case.start_date.isoformat() + 'Z' if case.start_date else None,
            'end_date': case.end_date.isoformat() + 'Z' if case.end_date else None,
            'documents_count': case.documents_count
        } for case in cases]

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_get_client_by_id',
            entity_type='client',
            entity_id=client.client_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'client_id': client_id},
            response_status='success'
        ).save()

        return success_response(
            message="Client details fetched",
            data={
                'client_id': client.client_id,
                'client_name': client.client_name,
                'cases': cases_data
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_get_client_by_id',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'client_id': client_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/admin/delete_client/<client_id>', methods=['DELETE'])
@admin_required
def admin_delete_client(client_id):
    """
    Delete Client (Admin Only)
    ---
    tags:
      - Admin - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: client_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Client deleted successfully
      404:
        description: Client not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client

        admin_user_id = get_jwt_identity()

        # Find client
        client = Client.objects(client_id=client_id, is_deleted=False).first()
        if not client:
            ActionLog(
                action_log_id=action_log_id,
                user_id=admin_user_id,
                action_type='admin_delete_client',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'client_id': client_id},
                response_status='failed',
                error_details={'reason': 'Client not found'}
            ).save()

            return error_response(
                message="Client not found or already deleted",
                error_code="CLIENT_NOT_FOUND",
                error_details="No client found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Soft delete
        client.is_deleted = True
        client.deleted_at = datetime.utcnow()
        client.deleted_by = admin_user_id
        client.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_delete_client',
            entity_type='client',
            entity_id=client.client_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'client_id': client_id},
            response_status='success'
        ).save()

        return success_response(
            message="Client deleted successfully",
            data={},
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_delete_client',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'client_id': client_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ==================== ANNOUNCEMENT MANAGEMENT (ADMIN) ====================

@api_v1_bp.route('/admin/create_announcement', methods=['POST'])
@admin_required
def admin_create_announcement():
    """
    Create Announcement (Admin Only)
    ---
    tags:
      - Admin - Announcements
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: body
        in: body
        required: true
        schema:
          type: object
          required:
            - title
            - message
          properties:
            title:
              type: string
              example: "New Feature Alert: Custom FAQ Training"
            message:
              type: string
              example: "You can now upload your own knowledge base..."
            status:
              type: string
              enum: [active, archived]
              default: active
    responses:
      201:
        description: Announcement created successfully
      400:
        description: Validation error
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.announcement import Announcement

        admin_user_id = get_jwt_identity()
        data = request.get_json()

        # Validate
        title = data.get('title', '').strip()
        message = data.get('message', '').strip()
        status = data.get('status', 'active').strip()

        if not title or not message:
            return error_response(
                message="Title and message are required",
                error_code="VALIDATION_ERROR",
                error_details="Both title and message fields are required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get admin name
        admin_user = User.objects(user_id=admin_user_id).first()
        admin_name = admin_user.full_name if admin_user else "Admin"

        # Create announcement
        new_announcement = Announcement(
            title=title,
            message=message,
            status=status,
            created_by=admin_user_id,
            created_by_name=admin_name,
            created_at=datetime.utcnow(),
            updated_at=datetime.utcnow()
        )
        new_announcement.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=admin_user_id,
            action_type='admin_create_announcement',
            entity_type='announcement',
            entity_id=new_announcement.announcement_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success'
        ).save()

        return success_response(
            message="Announcement created successfully",
            data={
                'announcement_id': new_announcement.announcement_id,
                'title': new_announcement.title,
                'message': new_announcement.message,
                'created_at': new_announcement.created_at.isoformat() + 'Z',
                'updated_at': new_announcement.updated_at.isoformat() + 'Z',
                'status': new_announcement.status,
                'created_by': admin_name
            },
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'admin_user_id' not in locals() else admin_user_id,
            action_type='admin_create_announcement',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )
