from mongoengine import Document, StringField, DateTimeField, DictField
from datetime import datetime
import uuid


class Notification(Document):
    """Notification model for user notifications"""
    meta = {'collection': 'notifications'}

    # Primary fields
    notification_id = StringField(required=True, unique=True, default=lambda: str(uuid.uuid4()))
    user_id = StringField(required=True, index=True)  # Recipient user

    # Notification content
    title = StringField(required=True)
    message = StringField(required=True)

    # Notification type and status
    type = StringField(
        required=True,
        choices=['success', 'info', 'warning', 'error'],
        default='info'
    )
    status = StringField(
        required=True,
        choices=['unread', 'read'],
        default='unread',
        index=True
    )

    # Related entities (optional)
    entity_type = StringField()  # e.g., 'case', 'document', 'user'
    entity_id = StringField()  # ID of related entity

    # Additional data
    metadata = DictField()  # Any additional notification data
    action_url = StringField()  # URL to navigate to when clicked

    # Timestamps
    created_at = DateTimeField(default=datetime.utcnow, index=True)
    read_at = DateTimeField()

    # Priority
    priority = StringField(
        choices=['low', 'normal', 'high', 'urgent'],
        default='normal'
    )

    def __repr__(self):
        return f'<Notification {self.notification_id}: {self.title}>'

    def mark_as_read(self):
        """Mark notification as read"""
        if self.status == 'unread':
            self.status = 'read'
            self.read_at = datetime.utcnow()
            self.save()

    @staticmethod
    def create_notification(user_id, title, message, notification_type='info',
                          entity_type=None, entity_id=None, metadata=None,
                          action_url=None, priority='normal'):
        """
        Create a new notification for a user

        Args:
            user_id: Target user ID
            title: Notification title
            message: Notification message
            notification_type: Type of notification (success, info, warning, error)
            entity_type: Related entity type (optional)
            entity_id: Related entity ID (optional)
            metadata: Additional data (optional)
            action_url: Action URL (optional)
            priority: Notification priority (optional)

        Returns:
            Notification: Created notification instance
        """
        notification = Notification(
            user_id=user_id,
            title=title,
            message=message,
            type=notification_type,
            entity_type=entity_type,
            entity_id=entity_id,
            metadata=metadata or {},
            action_url=action_url,
            priority=priority
        )
        notification.save()
        return notification

    @staticmethod
    def get_unread_count(user_id):
        """Get count of unread notifications for a user"""
        return Notification.objects(user_id=user_id, status='unread').count()

    @staticmethod
    def mark_all_as_read(user_id):
        """Mark all notifications as read for a user"""
        notifications = Notification.objects(user_id=user_id, status='unread')
        count = notifications.count()
        current_time = datetime.utcnow()

        for notification in notifications:
            notification.status = 'read'
            notification.read_at = current_time
            notification.save()

        return count
