from flask import Flask
from flask_cors import CORS
from flask_jwt_extended import JWTManager
from flask_bcrypt import Bcrypt
from mongoengine import connect
from config import config
from flasgger import Swagger
import os

# Initialize extensions
jwt = JWTManager()
bcrypt = Bcrypt()


def create_app(config_name=None):
    """Application factory pattern"""
    if config_name is None:
        config_name = os.getenv('FLASK_ENV', 'development')

    app = Flask(__name__)
    app.config.from_object(config[config_name])

    # Initialize MongoDB
    connect(host=app.config['MONGO_URI'])

    # Initialize extensions
    jwt.init_app(app)
    bcrypt.init_app(app)

    # Configure CORS with specific origins
    CORS(app,
         resources={
             r"/api/*": {"origins": app.config['CORS_ORIGINS']},
             r"/apispec.json": {"origins": app.config['CORS_ORIGINS']},
             r"/flasgger_static/*": {"origins": app.config['CORS_ORIGINS']}
         },
         supports_credentials=True,
         allow_headers=['Content-Type', 'Authorization', 'Accept', 'X-Requested-With'],
         expose_headers=['Content-Type', 'Authorization'],
         methods=['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS', 'PATCH'])

    # Configure Swagger
    swagger_config = {
        "headers": [],
        "specs": [
            {
                "endpoint": 'apispec',
                "route": '/apispec.json',
                "rule_filter": lambda rule: True,
                "model_filter": lambda tag: True,
            }
        ],
        "static_url_path": "/flasgger_static",
        "swagger_ui": True,
        "specs_route": "/api/docs"
    }

    # Get the API host from environment variable or use production domain
    api_host = os.getenv('API_HOST', 'imcode-backend.viewmysitenow.com')

    swagger_template = {
        "swagger": "2.0",
        "info": {
            "title": "imCode API Documentation",
            "description": "API documentation for imCode - Legal Case Management System",
            "version": "1.0.0",
            "contact": {
                "name": "imCode Support",
                "url": "https://imcode.viewmysitenow.com"
            }
        },
        "host": api_host,
        "basePath": "/api/v1",
        "schemes": ["https", "http"],
        "securityDefinitions": {
            "Bearer": {
                "type": "apiKey",
                "name": "Authorization",
                "in": "header",
                "description": "JWT Authorization header using the Bearer scheme. Example: 'Authorization: Bearer {token}'"
            }
        }
    }

    Swagger(app, config=swagger_config, template=swagger_template)

    # Register blueprints
    from app.api.v1 import api_v1_bp
    app.register_blueprint(api_v1_bp, url_prefix='/api/v1')

    # Register error handlers
    from app.utils.error_handlers import register_error_handlers
    register_error_handlers(app)

    return app
